//
//  TAWAccountDetailsViewController.swift
//  TAW
//
//  Created by Andrew Steven on 18/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit

class TAWAccountDetailsViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var backButton: UIButton!
    @IBOutlet weak var firstNameTextField: PBTextField!
    @IBOutlet weak var lastNameTextField: PBTextField!
    @IBOutlet weak var usernameTextField: PBTextField!
    @IBOutlet weak var editEmailButton: UIButton!
    @IBOutlet weak var editPasswordButton: UIButton!
    @IBOutlet weak var saveButton: PBButton!
    
    @IBOutlet weak var saveButtonHeight: NSLayoutConstraint!
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()

        let user = LocalStorage.shared.user

        firstNameTextField.theTextFieldLabel.text = "FIRST NAME"
        firstNameTextField.theTextField.text = user?.firstName
        firstNameTextField.setType(type: .standard, validation_type: .standard)
        firstNameTextField.theTextField.addTarget(self,
                                                  action: #selector(textFieldDidChange(textField:)),
                                                  for: .editingChanged)
        
        lastNameTextField.theTextFieldLabel.text = "LAST NAME"
        lastNameTextField.theTextField.text = user?.lastName
        lastNameTextField.setType(type: .standard, validation_type: .standard)
        lastNameTextField.theTextField.addTarget(self,
                                                 action: #selector(textFieldDidChange(textField:)),
                                                 for: .editingChanged)
        
        usernameTextField.theTextFieldLabel.text = "USERNAME"
        usernameTextField.theTextField.text = user?.username
        usernameTextField.setType(type: .standard, validation_type: .standard)
        usernameTextField.theTextField.addTarget(self,
                                                 action: #selector(textFieldDidChange(textField:)),
                                                 for: .editingChanged)
        
        let tapGesture = UITapGestureRecognizer(target: self,
                                                action: #selector(dismissKeyboard))
        tapGesture.delegate = self
        self.view.addGestureRecognizer(tapGesture)
        
        self.saveButton.buttonBackgroundColor = TAW.Colors.greyButtonColor
        self.saveButton.shaddowOppacity = 0
    }
    
    override func viewWillAppear(_ animated: Bool) {
        self.animations = self.isMovingToParent ? [.slide(.left, .slightly), .fadeIn] : [.slide(.right, .slightly), .fadeIn]
        super.viewWillAppear(animated)
    }

    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        self.setupView()
    }
    
    private func setupView() {
        
        if UIDevice.isSmallDevice {
            self.view.layoutIfNeeded()
            self.saveButtonHeight.constant = 45
            self.view.layoutIfNeeded()
        }

        if Connectivity.isConnectedToInternet {
            firstNameTextField.theTextField.isEnabled = true
            lastNameTextField.theTextField.isEnabled = true
            usernameTextField.theTextField.isEnabled = true
            saveButton.isEnabled = true
        } else {
            firstNameTextField.theTextField.isEnabled = false
            firstNameTextField.alpha = 0.3
            lastNameTextField.theTextField.isEnabled = false
            lastNameTextField.alpha = 0.3
            usernameTextField.theTextField.isEnabled = false
            usernameTextField.alpha = 0.3
            saveButton.isEnabled = false
            PBBannerView.shared.showBanner(inView: self.navigationController!.view, withTitle: "No internet connection", style: .noInternet)
        }
    }
    
    // MARK: - UITextField -
    
    @objc func textFieldDidChange(textField: UITextField){
        self.invisibleCheck()
    }
    
    @objc func dismissKeyboard() {
        self.view.endEditing(true)
    }
    
    // MARK: - Button actions -
    
    @IBAction func goBack() {
        self.navigationController?.popViewController(animated: true)
    }
    
    @IBAction func saveDetails() {
        if valid() {
            if LocalStorage.shared.user?.username != self.usernameTextField.theTextField.text {
                self.checkUsernameIsAvailable()
            } else {
                self.updateDetails()
            }
        }
    }
    
    // MARK: - Validation functions -
    
    private func invisibleCheck() {
        
        let user = LocalStorage.shared.user
        let firstName = user?.firstName
        let lastName = user?.lastName
        let username = user?.username
        
        if self.firstNameTextField.theTextField.text == firstName &&
            self.lastNameTextField.theTextField.text == lastName &&
            self.usernameTextField.theTextField.text == username {
            self.saveButton.buttonBackgroundColor = TAW.Colors.greyButtonColor
            self.saveButton.shaddowOppacity = 0
        } else {
            var isEnabled = false
            let validFirstName = self.firstNameTextField.invisibleCheckValidation(type: .standard)
            let validLastName = self.lastNameTextField.invisibleCheckValidation(type: .standard)
            let validUsername = self.usernameTextField.invisibleCheckValidation(type: .standard)
            
            if validFirstName && validLastName && validUsername {
                isEnabled = true
            }
            
            if isEnabled {
                self.saveButton.buttonBackgroundColor = TAW.Colors.redButtonColor
                self.saveButton.shaddowOppacity = 1
            } else {
                self.saveButton.buttonBackgroundColor = TAW.Colors.greyButtonColor
                self.saveButton.shaddowOppacity = 0
            }
        }
    }
    
    private func valid() -> Bool {
        
        let user = LocalStorage.shared.user
        let firstName = user?.firstName
        let lastName = user?.lastName
        let username = user?.username
        
        if self.firstNameTextField.theTextField.text == firstName &&
            self.lastNameTextField.theTextField.text == lastName &&
            self.usernameTextField.theTextField.text == username {
            self.showAlertForAPIError("You have not altered any details")
            return false
        } else {
            let validFirstName = self.firstNameTextField.checkValidation(required: true, type: .standard)
            if !validFirstName { return false }
            
            let validLastName = self.lastNameTextField.checkValidation(required: true, type: .standard)
            if !validLastName { return false }
            
            let validUsername = self.usernameTextField.checkValidation(required: true, type: .standard)
            if !validUsername { return false }
            
            return true
        }
    }
    
    // MARK: - API functions -
    
    private func checkUsernameIsAvailable() {
        
        let params = [
            "username": self.usernameTextField.theTextField.text!
        ]
        
        APIClient.checkUsernameIsAvailable(parameters: params) { (jsonResponse, error) in
            if jsonResponse["success"] as? Bool == true {
                self.updateDetails()
            } else {
                self.usernameTextField.overrideFailValidation(message: "Username already taken")
            }
        }
    }
    
    private func updateDetails() {
        
        let params = [
            "firstname": self.firstNameTextField.theTextField.text!,
            "lastname": self.lastNameTextField.theTextField.text!,
            "username": self.usernameTextField.theTextField.text!
        ]
        
        APIClient.updateUserDetails(parameters: params) { (jsonResponse, error) in
            if jsonResponse["success"] as? Bool == true {
                let user = LocalStorage.shared.user
                user?.firstName = self.firstNameTextField.theTextField.text!
                user?.lastName = self.lastNameTextField.theTextField.text!
                user?.username = self.usernameTextField.theTextField.text!
                LocalStorage.shared.user = user
                self.showAlertWithTitle("Success", message: jsonResponse["message"] as! String)
            } else {
                PBLoadingOverlay.shared.hideOverlay {
                    let message = jsonResponse["message"] as! String
                    if message == "Invalid Token" {
                        self.invalidateSession()
                    } else {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                    }
                }
            }
        }
    }
}
